///////////////////////////////////////////////////////////////////////////////
//
//    (C) Copyright 1995 - 2000 OSR Open Systems Resources, Inc.
//    All Rights Reserved
//
//    This sofware is supplied for instructional purposes only.
//
//    OSR Open Systems Resources, Inc. (OSR) expressly disclaims any warranty
//    for this software.  THIS SOFTWARE IS PROVIDED  "AS IS" WITHOUT WARRANTY
//    OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT LIMITATION,
//    THE IMPLIED WARRANTIES OF MECHANTABILITY OR FITNESS FOR A PARTICULAR
//    PURPOSE.  THE ENTIRE RISK ARISING FROM THE USE OF THIS SOFTWARE REMAINS
//    WITH YOU.  OSR's entire liability and your exclusive remedy shall not
//    exceed the price paid for this material.  In no event shall OSR or its
//    suppliers be liable for any damages whatsoever (including, without
//    limitation, damages for loss of business profit, business interruption,
//    loss of business information, or any other pecuniary loss) arising out
//    of the use or inability to use this software, even if OSR has been
//    advised of the possibility of such damages.  Because some states/
//    jurisdictions do not allow the exclusion or limitation of liability for
//    consequential or incidental damages, the above limitation may not apply
//    to you.
//
//    OSR Open Systems Resources, Inc.
//    105 Route 101A Suite 19
//    Amherst, NH 03031  (603) 595-6500 FAX: (603) 595-6503
//    email bugs to: bugs@osr.com
//
//
//    MODULE:
//
//        OSRNTAPI.C
//
//    ABSTRACT:
//
//      This file contains the skeleton code for a device driver
//      that illustrates some NT API pitfalls. 
//
//    AUTHOR(S):
//
//        OSR Open Systems Resources, Inc.
// 
//    REVISION:   
//
//
///////////////////////////////////////////////////////////////////////////////
#include "osrntapi.h"


//
// When USER_HANDLE is defined, the HANDLE
//  to the file will be opened in the context
//  of the calling process without the OBJ_KERNEL_HANDLE
//  flag being set, making it valid only in that process'
//  context.
//
#define USER_HANDLE 1

//
// The following pragma allows the DriverEntry code to be discarded once
// initialization is completed
//
#pragma alloc_text(INIT,DriverEntry)

///////////////////////////////////////////////////////////////////////////////
//
//  DriverEntry
//
//    This routine is called by NT when the driver is first loaded.  It is the
//    responsibility of this routine to find it's device and create whatever
//    device objects it needs.
//
//  INPUTS:
//
//      DriverObj - Address of the DRIVER_OBJECT created by NT for this driver.
//
//      RegistryPath - UNICODE_STRING which represents this drivers KEY in the
//                   Registry.  
//
//  OUTPUTS:
//
//      None.
//
//  RETURNS:
//
//      STATUS_SUCCESS. Otherwise an error indicating why the driver could not
//                    Load.
//
//  IRQL:
//
//    This routine is called at IRQL_PASSIVE_LEVEL.
//
//  NOTES:
//
//
///////////////////////////////////////////////////////////////////////////////
NTSTATUS
DriverEntry(PDRIVER_OBJECT DriverObj, PUNICODE_STRING RegistryPath)
{
    NTSTATUS            code;
    POSR_DEVICE_EXT     devExt;
    PDEVICE_OBJECT      devObj;
    UNICODE_STRING      devName;
    UNICODE_STRING      linkName;

    KdPrint(("\nOSR NTAPI Driver -- Compiled %s %s\n",__DATE__, __TIME__));

    //
    // Establish dispatch entry points for the functions we support
    //

    //
    // Note same function for create and close handling
    //
    DriverObj->MajorFunction[IRP_MJ_CREATE]         =  OsrCreateClose;
    DriverObj->MajorFunction[IRP_MJ_CLOSE]          =  OsrCreateClose;

    DriverObj->MajorFunction[IRP_MJ_DEVICE_CONTROL] =  OsrDeviceControl;
    DriverObj->MajorFunction[IRP_MJ_READ]           =  OsrRead;

    //
    // Unload function
    //
    DriverObj->DriverUnload = OsrUnload;

    //
    // Initialize the UNICODE device name.  This will be the "native NT" name
    // for our device.
    //
    RtlInitUnicodeString(&devName, OSRNTAPI_DEVICENAME);

    //
    // Ask the I/O Manager to create the device object and
    // device extension
    //
    code = IoCreateDevice(DriverObj,
                          sizeof(OSR_DEVICE_EXT),
                          &devName,
                          FILE_DEVICE_OSR,
                          0,       
                          FALSE,
                          &devObj);


    if(!NT_SUCCESS(code))  {

        KdPrint(("IoCreateDevice failed.  Status = 0x%0x\n", code));

        return(STATUS_UNSUCCESSFUL);
    }    

    //
    // Get a pointer to our device extension
    //
    devExt = (POSR_DEVICE_EXT)devObj->DeviceExtension;

    //
    // Save the device object pointer away for future reference... Not
    // that we ever reference it in the OSR driver.
    //
    devExt->DeviceObject = devObj;

    //
    // Initialize our work item event
    //
    KeInitializeEvent(&devExt->WorkItemEvent, NotificationEvent, FALSE);

    //
    // Next, make the device accessible from user-mode applications.
    // Note that this name can be either the same or different from
    // the native "kernel mode" name of the device object, given above.
    //
    RtlInitUnicodeString(&linkName, OSRNTAPI_LINKNAME);

    code = IoCreateSymbolicLink(&linkName, &devName);

    if (!NT_SUCCESS(code)) {

        KdPrint(("IoCreateSymbolicLink failed.  Status = 0x%x\n", code));

        IoDeleteDevice(devObj);

        //
        // Indicate load failure to the I/O manager; driver image is deleted...
        //
        return(code);
    }


    //
    // Tell the I/O Manger to buffer our reads/writes
    //
    devObj->Flags |= DO_BUFFERED_IO;


    KdPrint(("OsrDriverEntry: Leaving\n"));

    return(STATUS_SUCCESS);

}

///////////////////////////////////////////////////////////////////////////////
//
//  OsrCreateClose
//
//    This is the dispatch entry point for processing both IRP_MJ_CREATE
//    and IRP_MJ_CLOSE functions.  Since this is a simple device driver,
//    there's really nothing to do here by complete the requests with success.
//
//
//  INPUTS:
//
//    DeviceObject - Address of the DEVICE_OBJECT for our device.
//  
//    Irp - Address of the IRP representing the IRP_MJ_CREATE or IRP_MJ_CLOSE
//          call.
//
//  OUTPUTS:
//
//      None.
//
//  RETURNS:
//
//    STATUS_SUCCESS.   We never fail this function.        
//
//  IRQL:
//
//    This routine is called at IRQL_PASSIVE_LEVEL.
//
//  NOTES:
//
///////////////////////////////////////////////////////////////////////////////
NTSTATUS OsrCreateClose(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{

    NTSTATUS code;
    POSR_DEVICE_EXT devExt = (POSR_DEVICE_EXT)DeviceObject->DeviceExtension;
    IO_STATUS_BLOCK     iosb;

    KdPrint(("OsrCreateClose: Starting\n"));

    //
    // Nothing much to do....
    //
    Irp->IoStatus.Status = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    KdPrint(("OsrCreateClose: Leaving\n"));

    return(STATUS_SUCCESS);
}



///////////////////////////////////////////////////////////////////////////////
//
//  OsrDeviceControl
//
//    This is the dispatch entry point for processing IRP_MJ_DEVICE_CONTROL
//    Irps.  
//
//
//  INPUTS:
//
//    DeviceObject - Address of the DEVICE_OBJECT for our device.
//  
//    Irp - Address of the IRP representing the IRP_MJ_DEVICE_CONTROL call.
//
//  OUTPUTS:
//
//      None.
//
//  RETURNS:
//
//    STATUS_SUCCESS if success, otherwise an appropriate status is returned.
//
//  IRQL:
//
//    This routine is called at IRQL_PASSIVE_LEVEL.
//
//  NOTES:
//
//    We presently only support IOCTL_OSR_OSR.
//
///////////////////////////////////////////////////////////////////////////////
NTSTATUS OsrDeviceControl(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
    ULONG              operation;
    PIO_STACK_LOCATION ioStack = IoGetCurrentIrpStackLocation(Irp);
    NTSTATUS           code;
    POSR_DEVICE_EXT    devExt = (POSR_DEVICE_EXT)DeviceObject->DeviceExtension;
    IO_STATUS_BLOCK    iosb;
    OBJECT_ATTRIBUTES  oa;
    UNICODE_STRING     logFileName;

    KdPrint(("OsrDeviceControl: Starting\n"));

    operation = ioStack->Parameters.DeviceIoControl.IoControlCode;

    //
    // Set the status field to zero for completion
    //
    Irp->IoStatus.Information = 0;

    switch (operation) {

        case IOCTL_OSR_ENABLE_LOGGING:

            if (!devExt->LoggingEnabled) {

                RtlInitUnicodeString(&logFileName, OSRNTAPI_LOGFILE);

#ifdef USER_HANDLE

                //
                // If USER_HANDLE is defined, do not specify
                //  OBJ_KERNEL_HANDLE in the OBJECT_ATTRIBUTES. 
                //  This causes this HANDLE to be specific to
                //  this process context.
                //
                InitializeObjectAttributes(&oa, 
                                           &logFileName,
                                           OBJ_CASE_INSENSITIVE,
                                           NULL,
                                           NULL);
#else 

                //
                // Otherwise, specify OBJ_KERNEL_HANDLE and make this
                //  HANDLE valid in all process contexts.
                //
                InitializeObjectAttributes(&oa, 
                                           &logFileName,
                                           OBJ_CASE_INSENSITIVE|OBJ_KERNEL_HANDLE,
                                           NULL,
                                           NULL);
#endif

                //
                // Create our log file with write access
                //  and also make all operations on the file
                //  synchronous.
                //
                code = ZwCreateFile(&devExt->LogFileHandle,
                                    GENERIC_WRITE | SYNCHRONIZE,
                                    &oa,
                                    &iosb,
                                    NULL,
                                    FILE_ATTRIBUTE_NORMAL,
                                    0,
                                    FILE_OVERWRITE_IF,
                                    FILE_NON_DIRECTORY_FILE | FILE_SYNCHRONOUS_IO_NONALERT,
                                    NULL,
                                    0);

                if (!NT_SUCCESS(code)) {

                    KdPrint(("OsrDeviceControl: ZwCreateFile failed! Returned 0x%x!\n", code));

                    devExt->LoggingEnabled = FALSE;

                } else {

                    KdPrint(("OsrDeviceControl: Log file %ls created!\n", OSRNTAPI_LOGFILE));

                    devExt->LoggingEnabled = TRUE;

                }

            } else {

                code = STATUS_UNSUCCESSFUL;

            }

            KdPrint(("OsrDeviceControl: IOCTL_OSR_ENABLE_LOGGING processed\n"));

            break;

        case IOCTL_OSR_DISABLE_LOGGING:

            if (devExt->LoggingEnabled) {

                code = ZwClose(devExt->LogFileHandle);

                //
                // Either we're running in the same process
                //  context that we created the HANDLE in
                //  or the HANDLE is good in all contexts,
                //  so this should always succeed
                //
                ASSERT(code == STATUS_SUCCESS);

                devExt->LoggingEnabled = FALSE;


            } else {

                code = STATUS_UNSUCCESSFUL;

            }

            KdPrint(("OsrDeviceControl: IOCTL_OSR_DISABLE_LOGGING processed\n"));

            break;


        case IOCTL_OSR_EXECUTE_WORK_ITEM:

            //
            // Try to allocate a work item for 
            //  us to queue
            //
            devExt->WorkItem = IoAllocateWorkItem(DeviceObject);

            if (devExt->WorkItem != NULL) {

                //
                // Queue the work item and wait for it to
                //  complete...
                //
                IoQueueWorkItem(devExt->WorkItem,
                                OsrWorkItem,
                                DelayedWorkQueue,
                                devExt);

                code = KeWaitForSingleObject(&devExt->WorkItemEvent,
                                             Executive,
                                             KernelMode,
                                             FALSE,
                                             NULL);

                if (!NT_SUCCESS(code)) {

                    KdPrint(("KeWaitForSingleObject failed! Returned 0x%x!\n", code));

                }

            } else {

                //
                // System is low on resources, nothing
                //  we can do but fail the request...
                //
                code = STATUS_INSUFFICIENT_RESOURCES;

            }

            KdPrint(("OsrDeviceControl: IOCTL_OSR_EXECUTE_WORK_ITEM processed\n"));

            break;



        default:

            //
            // This is the case for a bogus request code.
            //
            code = STATUS_INVALID_PARAMETER;

            KdPrint(("OsrDeviceControl: Invalid IOCTL code 0x%0x\n", operation));

            break;
    }

    //
    // Complete the I/O Request
    //
    Irp->IoStatus.Status = code;

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    KdPrint(("OsrDeviceControl: Leaving\n"));

    return(code);
}




///////////////////////////////////////////////////////////////////////////////
//
//  OsrRead
//
//    This is the Read dispatch entry point for the driver, called when the
//    I/O Manager has an IRP_MJ_READ request for the driver to process.
//
//  INPUTS:
//
//    DeviceObject - Address of the DEVICE_OBJECT for our device.
//  
//    Irp - Address of the IRP representing the IRP_MJ_READ call.
//
//  OUTPUTS:
//
//      None.
//
//  RETURNS:
//
//      STATUS_SUCCESS since we aren't really trying to read
//       from anything!
//
//  IRQL:
//
//    This routine is called at IRQL_PASSIVE_LEVEL.
//
//  NOTES:
//
//
///////////////////////////////////////////////////////////////////////////////
NTSTATUS OsrRead(PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
    PIO_STACK_LOCATION ioStack = IoGetCurrentIrpStackLocation(Irp);
    NTSTATUS           code;
    POSR_DEVICE_EXT    devExt = (POSR_DEVICE_EXT)DeviceObject->DeviceExtension;
    IO_STATUS_BLOCK    iosb;

    KdPrint(("OsrRead: Starting\n"));

    if (devExt->LoggingEnabled) {

        code = NtWriteFile(devExt->LogFileHandle,
                           NULL,
                           NULL,
                           NULL,
                           &iosb,
                           (PVOID)"OsrRead: NtWriteFile\r\n",
                           strlen("OsrRead: NtWriteFile\r\n"),
                           NULL,
                           NULL);

#ifdef USER_HANDLE

        //
        // If USER_HANDLE is defined, then we have a HANDLE
        //  value that is valid only in the process context 
        //  of our application. Right now we are running 
        //  in that process context (so the HANDLE is valid), 
        //  with our requestor mode set to USER. 
        //
        // Since we didn't go through ZwWriteFile, the attempt to 
        //  write was treated a USER request. NtWriteFile 
        //  attempted to validate our buffers by probing them 
        //  using ProbeForRead/ProbeForWrite, but the catch 
        //  here is that our buffers are in  KERNEL space. 
        //  ProbeForXxx always raise an exception 
        //  on kernel addresses and so we're going to get 
        //  STATUS_ACCESS_VIOLATIOn returned to us
        //
        ASSERT(code == STATUS_ACCESS_VIOLATION);

#else

        //
        // OBJ_KERNEL_HANDLE was used, but we're 
        //  going to get returned STATUS_INVALID_HANDLE!
        //
        // Remember, we're running with our requestor
        //  mode set to USER, and directly calling
        //  NtWriteFile. Since kernel HANDLEs aren't
        //  accessible to user mode programs, NtWriteFile
        //  returns telling us we have an invalid HANDLE
        //
        ASSERT(code == STATUS_INVALID_HANDLE);

#endif

        code = ZwWriteFile(devExt->LogFileHandle,
                           NULL,
                           NULL,
                           NULL,
                           &iosb,
                           (PVOID)"OsrRead: ZwWriteFile\r\n",
                           strlen("OsrRead: ZwWriteFile\r\n"),
                           NULL,
                           NULL);

        //
        // In this case, OBJ_KERNEL_HANDLE doesn't 
        //  matter. 
        //
        // If it wasn't specified, we're in the 
        //  same context that we created the HANDLE in so the HANDLE
        //  lookup succeeds. 
        //
        // If it was, we're going through ZwWriteFile 
        //  so our requestor mode gets properly set to 
        //  KERNEL and we're granted access to the HANDLE
        //
        ASSERT(code == STATUS_SUCCESS);

    }
    //
    // Complete the IRP with success
    //
    Irp->IoStatus.Status = STATUS_SUCCESS;
    Irp->IoStatus.Information =  ioStack->Parameters.Read.Length;

    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    KdPrint(("OsrRead: Leaving\n"));

    return(STATUS_SUCCESS);
}


///////////////////////////////////////////////////////////////////////////////
//
//  OsrWorkItem
//
//    This is the our work item that is scheduled when processing
//     IOCTL_OSR_EXECUTE_WORK_ITEM
//  
//
//  INPUTS:
//
//    DeviceObject - Address of the DEVICE_OBJECT for our device.
//  
//    Context - Address of our OSR_DEVICE_EXT
//
//  OUTPUTS:
//
//      None.
//
//  RETURNS:
//
//      None
//
//  IRQL:
//
//    This routine is called at IRQL_PASSIVE_LEVEL.
//
//  NOTES:
//
//
///////////////////////////////////////////////////////////////////////////////
VOID OsrWorkItem(PDEVICE_OBJECT DeviceObject, PVOID Context) 
{

    POSR_DEVICE_EXT devExt = (POSR_DEVICE_EXT)Context;
    NTSTATUS code;
    IO_STATUS_BLOCK iosb;

    KdPrint(("OsrWorkItem: Starting\n"));

    if (devExt->LoggingEnabled) {

        code = NtWriteFile(devExt->LogFileHandle,
                           NULL,
                           NULL,
                           NULL,
                           &iosb,
                           (PVOID)"OsrWorkItem: NtWriteFile\r\n",
                           strlen("OsrWorkItem: NtWriteFile\r\n"),
                           NULL,
                           NULL);

#ifdef USER_HANDLE

        //
        // We're running in the SYSTEM process context, 
        //  with our requestor mode set to KERNEL. If
        //  OBJ_KERNEL_HANDLE wasn't specified then 
        //  the lookup of the HANDLE in the SYSTEM process'
        //  handle table is going to FAIL, because it 
        //  is only valid in the application's context
        //
        ASSERT(code == STATUS_INVALID_HANDLE);

#else 
        //
        // If OBJ_KERNEL_HANDLE was specified, then
        //  this call is going to succeed beacuse our requestor
        //  mode is KERNEL and our HANDLE is a kernel 
        //  HANDLE.
        //
        ASSERT(code == STATUS_SUCCESS);

#endif

        code = ZwWriteFile(devExt->LogFileHandle,
                           NULL,
                           NULL,
                           NULL,
                           &iosb,
                           (PVOID)"OsrWorkItem: ZwWriteFile\r\n",
                           strlen("OsrWorkItem: ZwWriteFile\r\n"),
                           NULL,
                           NULL);

#ifdef USER_HANDLE

        //
        // This call is also going to fail with INVALID_HANDLE
        //  for the same reason taht NtWriteFile failed above,
        //  the HANDLE that we have is good only in the context of our
        //  application
        //
        ASSERT(code == STATUS_INVALID_HANDLE);

#else 

        //
        // If OBJ_KERNEL_HANDLE was specified, then
        //  this call is going to succeed beacuse our requestor
        //  mode is KERNEL and our HANDLE is a kernel 
        //  HANDLE.
        //
        ASSERT(code == STATUS_SUCCESS);

#endif
 
    }

    //
    // Free our work item
    //
    IoFreeWorkItem(devExt->WorkItem);
    
    //
    // And set our completion event
    //
    KeSetEvent(&devExt->WorkItemEvent, 0, FALSE);
    
    KdPrint(("OsrWorkItem: Leaving\n"));

    return;

}




///////////////////////////////////////////////////////////////////////////////
//
//  OsrUnload
//
//    This routine is our dynamic unload entry point.  We are called here when
//    the OS wants to unload our driver.  It is our responsibility to release any
//    resources we allocated.
//
//  INPUTS:
//
//      DriverObj - Address of our DRIVER_OBJECT.
//
//  OUTPUTS:
//
//      None.
//
//  RETURNS:
//
//      None.
//
//  IRQL:
//
//    This routine is called at IRQL_PASSIVE_LEVEL.
//
//  NOTES:
//
//    No doubt we pool leak at this entry point by not properly returning everything.
//
///////////////////////////////////////////////////////////////////////////////
VOID
OsrUnload(PDRIVER_OBJECT DriverObject)
{
    PDEVICE_OBJECT    devObj;
    UNICODE_STRING    linkName;
    POSR_DEVICE_EXT   devExt;
    NTSTATUS          code;
    IO_STATUS_BLOCK   iosb;
    OBJECT_ATTRIBUTES oa;
    UNICODE_STRING    logFileName;
           
    KdPrint(("OsrUnload: Starting\n"));

    //
    // For THIS driver, there will only ever be a single device object.
    // Because of this, we just get it from the DriverObj.  If this were
    // a multiple device driver, we would do this in a while loop...
    //
    devObj = DriverObject->DeviceObject;

    if (!devObj) {

        return;

    }

    devExt = (POSR_DEVICE_EXT)devObj->DeviceExtension;
    
    if (devExt->LoggingEnabled) {

        code = NtWriteFile(devExt->LogFileHandle,
                           NULL,
                           NULL,
                           NULL,
                           &iosb,
                           (PVOID)"OsrUnload: NtWriteFile\r\n",
                           strlen("OsrUnload: NtWriteFile\r\n"),
                           NULL,
                           NULL);

#ifdef USER_HANDLE

        //
        // We're again running in the SYSTEM process context,
        //  so we're going to fail here for the same reason
        //  we failed in OsrWorkItem
        //
        ASSERT(code == STATUS_INVALID_HANDLE);

#else 
        //
        // OBJ_KERNEL_HANDLE was specified and this is a
        //  a KERNEL request, so our context doesn't matter.and
        //  the call succeeds
        //
        ASSERT(code == STATUS_SUCCESS);

#endif

        code = ZwWriteFile(devExt->LogFileHandle,
                           NULL,
                           NULL,
                           NULL,
                           &iosb,
                           (PVOID)"OsrUnload: ZwWriteFile\r\n",
                           strlen("OsrUnload: ZwWriteFile\r\n"),
                           NULL,
                           NULL);

#ifdef USER_HANDLE

        //
        // This call is also going to fail with INVALID_HANDLE
        //  for the same reason taht NtWriteFile failed above
        //
        ASSERT(code == STATUS_INVALID_HANDLE);

#else 

        //
        // OBJ_KERNEL_HANDLE was specified and this is a
        //  a KERNEL request, so our context doesn't matter.and
        //  this call succeeds
        //
        ASSERT(code == STATUS_SUCCESS);

#endif
        
        code = ZwClose(devExt->LogFileHandle);

#ifdef USER_HANDLE

        //
        // This again fails because we're 
        //  in the wrong process context
        //
        ASSERT(code == STATUS_INVALID_HANDLE);

#else 

        //
        // Our HANDLE is good in any process
        //  context, so this will always succeed.
        //
        ASSERT(code == STATUS_SUCCESS);

#endif

        devExt->LogFileHandle = FALSE;
        
    }

    RtlInitUnicodeString(&linkName, OSRNTAPI_LINKNAME);

    IoDeleteSymbolicLink(&linkName);

    //
    // Delete the device object
    //
    IoDeleteDevice(devObj);

    KdPrint(("OsrUnload: Leaving\n"));

}

